﻿Imports System
Imports System.IO
Imports System.Reflection

'
'When an instance of this class is created,
'its constructor is passed a string indicating the
'type of point of sale system that the object will be a
'factory for. It gets a ConstructorInfo object
'that it will use to construct objects and make the Constructor
'object the value of the factoryConstructor instance variable. 
'
Public Class RecordFactoryFactory
    Private factoryConstructor As ConstructorInfo

    ' POS Types
    Public Enum PosTypes
        ABC
    End Enum

    '
    'constructor
    '
    'posType - The type of point of sale system that this object
    'will be creating IJournalRecordFactory object for.
    '
    Public Sub New(ByVal posType As PosTypes)
        Dim args() As Type = {GetType(StreamReader)}
        Dim factoryClass As Type
        If posType = PosTypes.ABC Then
            factoryClass = GetType(ABCJournalRecordFactory)
        Else
            Dim msg As String = "Unknown POS type: " & posType.ToString
            Throw New ApplicationException(msg)
        End If
        Try
            factoryConstructor = factoryClass.GetConstructor(args)
        Catch e As Exception
            Dim msg As String = "Error while constructing factory"
            Throw New ApplicationException(msg, e)
        End Try
    End Sub

    '
    'This method creates IJournalRecordFactory objects
    '
    Public Function CreateFactory(ByVal input As StreamReader) _
                    As IJournalRecordFactory
        Dim args() As Object = {input}
        Dim factory As Object
        Try
            factory = factoryConstructor.Invoke(args)
        Catch e As Exception
            Dim msg As String = "Error creating factory"
            Throw New ApplicationException(msg, e)
        End Try
        Return CType(factory, IJournalRecordFactory)
    End Function

End Class 'RecordFactoryFactory

' Objects that encapsulate the contents of a journal record 
' implement this interface.
Public Interface IJournalRecord
End Interface

'
'This interface is implemented by classes that are
'responsible for creating objects that encapsulate the
'contents of a point of sale journal file record.
'
Public Interface IJournalRecordFactory
    '
    'Return an object that encasulates the next record in a
    'journal file.
    '
    Function NextRecord() As IJournalRecord
End Interface

'
'This class implements the IJournalRecordFactory
'interface for a type of point of sale system named "ABC".
'
Public Class ABCJournalRecordFactory
    Implements IJournalRecordFactory

    ' Record Types
    Private Shared SALE_LINE_ITEM As String = "17"
    Private Shared START_OF_SALE As String = "4"

    Private input As StreamReader

    ' Counter for sequence number.
    Private sequenceNumber As Integer = 0


    Sub New(ByVal i As StreamReader)
        input = i
    End Sub

    '
    ' Return an object that encasulates the next record in a
    ' journal file.
    '
    Public Function NextRecord() As IJournalRecord _
      Implements IJournalRecordFactory.NextRecord
        Dim record As String = input.ReadLine()
        Dim tokens As String() = record.Split(New Char() {","c})
        sequenceNumber += 1

        If tokens(0) = START_OF_SALE Then
            Return CType(constructStartOfSale(tokens), _
                         IJournalRecord)
        ElseIf tokens(0) = SALE_LINE_ITEM Then
            Return CType(constructSaleLineItem(tokens), _
                         IJournalRecord)
        End If
        Throw New IOException("Unknown record type")
    End Function 'NextRecord

    Private _
    Function constructStartOfSale(ByVal tok() As String) _
             As StartOfSale
        Dim index As Integer = 1
        Dim transactionID As String = tok(index)
        index += 1 ' Move past ID
        index += 1 ' Skip mode indicator.
        Dim timestamp As DateTime = DateTime.Parse(tok(index))
        index += 1 ' Move past DateTime
        Dim terminalID As String = tok(index)
        Return New StartOfSale(terminalID, sequenceNumber, _
                               timestamp, transactionID)
    End Function 'constructStartOfSale

    Private _
    Function constructSaleLineItem(ByVal tok() As String) _
             As SaleLineItem
        Dim index As Integer = 1
        Dim transactionID As String = tok(index)
        index += 1 ' Move past ID
        index += 1 ' Skip mode indicator.
        Dim timestamp As DateTime = DateTime.Parse(tok(index))
        index += 1 ' Move past DateTime
        Dim terminalID As String = tok(index)
        Return New SaleLineItem(terminalID, sequenceNumber, _
                                timestamp, transactionID)
    End Function 'constructSaleLineItem

End Class 'ABCJournalRecordFactory




Class StartOfSale

    Public Sub New(ByVal id As String, ByVal num As Integer, ByVal time As DateTime, ByVal tid As String)
    End Sub 'New 
End Class 'StartOfSale
 _
Class SaleLineItem

    Public Sub New(ByVal id As String, ByVal num As Integer, ByVal time As DateTime, ByVal tid As String)
    End Sub 'New 
End Class 'SaleLineItem